// Variables to store extension state
let isActive = false;
let selectedBaseUrl = '';
let capturedRequests = [];

// Initialize extension state
chrome.runtime.onInstalled.addListener(() => {
  chrome.storage.local.set({
    isActive: false,
    selectedBaseUrl: '',
    capturedRequests: []
  });
});

// Listen for messages from popup
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === 'activate') {
    // Activate the extension with provided base URL
    selectedBaseUrl = message.selectedBaseUrl;
    isActive = true;
    capturedRequests = [];
    
    // Store state
    chrome.storage.local.set({
      isActive,
      selectedBaseUrl,
      capturedRequests
    });
    
    // Enable request tracking
    setupWebRequestListener();
    sendResponse({ success: true, message: 'Extension activated' });
  } 
  else if (message.action === 'deactivate') {
    // Deactivate the extension and clear data
    isActive = false;
    selectedBaseUrl = '';
    
    // Store state
    chrome.storage.local.set({
      isActive,
      selectedBaseUrl
    });
    
    sendResponse({ success: true, message: 'Extension deactivated' });
  }
  else if (message.action === 'getState') {
    // Return current state
    sendResponse({
      isActive,
      selectedBaseUrl,
      capturedRequests: capturedRequests.length
    });
  }
  else if (message.action === 'downloadData') {
    // Download captured data
    downloadCapturedData();
    sendResponse({ success: true, message: 'Download initiated' });
  }
  
  return true; // Keep the message channel open for async response
});

// Set up web request listener
function setupWebRequestListener() {
  if (chrome.webRequest.onBeforeRequest.hasListener(handleRequest)) {
    chrome.webRequest.onBeforeRequest.removeListener(handleRequest);
  }
  
  chrome.webRequest.onBeforeRequest.addListener(
    handleRequest,
    { urls: ["<all_urls>"] },
    ["requestBody"]
  );
  
  if (chrome.webRequest.onSendHeaders.hasListener(handleSendHeaders)) {
    chrome.webRequest.onSendHeaders.removeListener(handleSendHeaders);
  }
  
  chrome.webRequest.onSendHeaders.addListener(
    handleSendHeaders,
    { urls: ["<all_urls>"] },
    ["requestHeaders"]
  );
  
  if (chrome.webRequest.onCompleted.hasListener(handleCompleted)) {
    chrome.webRequest.onCompleted.removeListener(handleCompleted);
  }
  
  chrome.webRequest.onCompleted.addListener(
    handleCompleted,
    { urls: ["<all_urls>"] }
  );
}

/**
 * Enhanced URL endpoint classifier using multiple strategies
 * @param {string} url - The full URL to classify
 * @return {Object} The classified endpoint information
 */
function enhancedEndpointClassifier(url) {
  try {
    // Parse the URL
    const urlObj = new URL(url);
    const pathname = urlObj.pathname;
    
    // Split the path into segments
    const segments = pathname.split('/').filter(segment => segment);
    const originalSegments = [...segments];
    
    // Collection of potential ID segments with confidence scores
    const potentialIds = [];
    
    // Step 1: Apply pattern recognition to identify common ID formats
    const idPatterns = [
      { pattern: /^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i, weight: 1.0, type: 'uuid' }, // UUID
      { pattern: /^\d{1,10}$/, weight: 0.7, type: 'numeric' }, // Simple numeric ID (1-10 digits)
      { pattern: /^[a-f0-9]{24}$/i, weight: 0.9, type: 'mongo' }, // MongoDB ObjectId
      { pattern: /^[A-Za-z0-9_-]{21,22}$/i, weight: 0.8, type: 'firebase' }, // Firebase ID
      { pattern: /^[a-f0-9]{32}$/i, weight: 0.8, type: 'md5' }, // MD5 hash
      { pattern: /^[a-f0-9]{7,40}$/i, weight: 0.6, type: 'hexadecimal' }, // Generic hex string (like Git commit)
      { pattern: /^[A-Za-z0-9_-]{4,}={0,2}$/, weight: 0.75, type: 'base64url' } // Base64 URL encoded string
    ];
    
    // Step 2: Analyze the path structure based on common REST conventions
    // Most REST APIs follow patterns like /resource/{id}/subresource/{id}
    const restPositionalAnalysis = segments.map((segment, index) => {
      // Most APIs have alternating resource/id patterns
      const isLikelyId = index % 2 === 1;
      
      // Check if segment matches common ID formats
      let patternScore = 0;
      let matchType = null;
      
      idPatterns.forEach(({ pattern, weight, type }) => {
        if (pattern.test(segment)) {
          patternScore = Math.max(patternScore, weight);
          matchType = type;
        }
      });
      
      // Return segment analysis with confidence score
      return {
        segment,
        index,
        patternScore,
        restPositionScore: isLikelyId ? 0.5 : 0,
        matchType,
        isSpecialFormat: segment.includes('-') || segment.length > 10, // Typical IDs have special formats or are long
        totalConfidence: patternScore + (isLikelyId ? 0.5 : 0)
      };
    });
    
    // Step 3: Make decision about each segment
    const classifiedSegments = restPositionalAnalysis.map(analysis => {
      const { segment, totalConfidence } = analysis;
      
      // High confidence threshold for ID classification
      const IS_ID_THRESHOLD = 0.7;
      
      // If confidence is high, replace with parameter placeholder
      if (totalConfidence >= IS_ID_THRESHOLD) {
        potentialIds.push({
          segment,
          replacedWith: '{id}',
          confidence: totalConfidence,
          position: analysis.index,
          type: analysis.matchType || 'unknown'
        });
        return '{id}';
      }
      
      // Otherwise, leave as-is
      return segment;
    });
    
    // Reconstruct the path with classified segments
    const classifiedPath = '/' + classifiedSegments.join('/');
    
    // Include query parameters if the original URL had them
    const queryParams = urlObj.search ? 
      urlObj.search.replace(/=[^&]+/g, '={value}') : 
      '';
    
    return {
      originalUrl: url,
      classifiedEndpoint: classifiedPath + queryParams,
      confidence: potentialIds.length > 0 ? 
        potentialIds.reduce((sum, id) => sum + id.confidence, 0) / potentialIds.length : 
        1.0, // If no IDs found, path is already classified
      identifiedIds: potentialIds
    };
  } catch (error) {
    console.error('Error classifying endpoint:', error);
    return { 
      originalUrl: url,
      classifiedEndpoint: url,
      confidence: 0,
      error: error.message
    };
  }
}

// Temporary storage for request data
const requestData = {};

// Handle request before it's sent
function handleRequest(details) {
  if (!isActive || !selectedBaseUrl) return;
  
  // Check if URL contains the selected base URL
  if (details.url.includes(selectedBaseUrl)) {
    // Classify the endpoint using the enhanced classifier
    const classificationResult = enhancedEndpointClassifier(details.url);
    
    // Initialize request data object
    requestData[details.requestId] = {
      url: details.url,
      method: details.method,
      timestamp: new Date().toISOString(),
      derivedPath: classificationResult.classifiedEndpoint,
      derivedPathConfidence: classificationResult.confidence,
      requestBody: null
    };
    
    // Process request body if it exists
    if (details.requestBody) {
      try {
        // Handle raw data
        if (details.requestBody.raw && details.requestBody.raw.length > 0) {
          // Get the raw data
          const rawData = details.requestBody.raw[0];
          
          if (rawData.bytes) {
            // Convert ArrayBuffer to string
            const decoder = new TextDecoder('utf-8');
            const rawString = decoder.decode(rawData.bytes);
            
            try {
              // Try to parse as JSON
              const jsonData = JSON.parse(rawString);
              requestData[details.requestId].requestBody = jsonData;
            } catch (e) {
              // If not valid JSON, store as string
              requestData[details.requestId].requestBody = rawString;
            }
          } else {
            requestData[details.requestId].requestBody = '[Raw data unavailable]';
          }
        } 
        // Handle form data
        else if (details.requestBody.formData) {
          requestData[details.requestId].requestBody = details.requestBody.formData;
        }
      } catch (error) {
        console.error('Error processing request body:', error);
        requestData[details.requestId].requestBody = '[Error processing body]';
      }
    }
  }
}

// Handle request headers
function handleSendHeaders(details) {
  if (!isActive || !requestData[details.requestId]) return;
  
  if (requestData[details.requestId]) {
    requestData[details.requestId].requestHeaders = details.requestHeaders.reduce((headers, header) => {
      headers[header.name] = header.value;
      return headers;
    }, {});
  }
}

// Function to calculate the size of captured data in MB
function calculateDataSize() {
  const jsonString = JSON.stringify(capturedRequests);
  // Convert to MB (1 character in UTF-8 is typically 1-4 bytes, we'll use 2 as average)
  return (jsonString.length * 2) / (1024 * 1024);
}

// Function to show notification when data size reaches threshold
function showSizeNotification() {
  chrome.notifications.create({
    type: 'basic',
    iconUrl: 'assets/icons/icon48.png',
    title: 'ApisecBolt Data Size Alert',
    message: 'Captured data has reached 8MB. Consider downloading to free up space.',
    priority: 2,
    buttons: [
      { title: 'Download Now' }
    ]
  });
}

// Handle completed requests
function handleCompleted(details) {
  if (!isActive || !requestData[details.requestId]) return;
  
  if (requestData[details.requestId]) {
    requestData[details.requestId].statusCode = details.statusCode;
    
    // Add completed request to captured requests
    capturedRequests.push(requestData[details.requestId]);
    
    // Save to storage
    chrome.storage.local.set({ capturedRequests });
    
    // Check data size and notify if it exceeds 8MB
    const currentSize = calculateDataSize();
    if (currentSize >= 8) {
      showSizeNotification();
    }
    
    // Clean up
    delete requestData[details.requestId];
  }
}

// Function to download captured data
function downloadCapturedData() {
  if (capturedRequests.length === 0) {
    console.log("No data to download.");
    return;
  }

  // Add a derived endpoint summary
  const endpointSummary = {};
  capturedRequests.forEach(request => {
    if (request.derivedPath) {
      const key = `${request.method} ${request.derivedPath}`;
      if (!endpointSummary[key]) {
        endpointSummary[key] = { 
          count: 0, 
          examples: [] 
        };
      }
      endpointSummary[key].count++;
      if (endpointSummary[key].examples.length < 2) {
        endpointSummary[key].examples.push(request.url);
      }
    }
  });
  
  // Export both the detailed requests and a summary
  const exportData = {
    requests: capturedRequests,
    summary: {
      captureTime: new Date().toISOString(),
      totalRequests: capturedRequests.length,
      uniqueEndpoints: Object.keys(endpointSummary).length,
      endpoints: endpointSummary
    }
  };

  const jsonString = JSON.stringify(exportData, null, 2);
  const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
  const filename = `apisecbolt-capture-${timestamp}.json`;
  
  // Convert the string to a Data URL
  const dataUrl = 'data:application/json;base64,' + btoa(unescape(encodeURIComponent(jsonString)));
  
  // Use chrome.downloads.download with the Data URL
  chrome.downloads.download({
    url: dataUrl,
    filename: filename,
    saveAs: true
  }, (downloadId) => {
    if (chrome.runtime.lastError) {
      console.error("Download failed:", chrome.runtime.lastError);
    } else {
      console.log("Download started with ID:", downloadId);
    }
  });
}

// Add notification click handler
chrome.notifications.onButtonClicked.addListener((notificationId, buttonIndex) => {
  if (buttonIndex === 0) { // "Download Now" button
    downloadCapturedData();
  }
}); 